/*
Copyright (C) 2024 Centro de Computacao Cientifica e Software Livre
Departamento de Informatica - Universidade Federal do Parana - C3SL/UFPR

This file is part of simcaq-node.

simcaq-node is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

simcaq-node is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with simcaq-node.  If not, see <https://www.gnu.org/licenses/>.
*/

const express = require('express');

const PopulationOutOfSchoolApp = express.Router();

const libs = `${process.cwd()}/libs`;

const squel = require('squel');

const query = require(`${libs}/middlewares/query`).query;

const response = require(`${libs}/middlewares/response`);

const ReqQueryFields = require(`${libs}/middlewares/reqQueryFields`);

const id2str = require(`${libs}/middlewares/id2str`);

const config = require(`${libs}/config`); 

const cache = require('apicache').options({ debug: config.debug, statusCodes: {include: [200]} }).middleware;

let rqf = new ReqQueryFields();

const multiQuery = require(`${libs}/middlewares/multiQuery`);

PopulationOutOfSchoolApp.use(cache('15 day'));

PopulationOutOfSchoolApp.get('/years', (req, res, next) => {
    req.sql.from('pnad_novo')
    .field('DISTINCT pnad_novo.ano_ref', 'year')
    .where('pnad_novo.ano_ref >= 2019')
    next();
}, query, response('years'));

PopulationOutOfSchoolApp.get('/illiteracy', (req, res, next) => {
    req.result = []

    for (let i = 0; i < 2; i++) {
        req.result.push({
            id: i, name: id2str.illiteracy(i)
        });

    }
    req.result.push({id: 9, name: id2str.illiteracy(9)});
    next();
}, response('illiteracy'));

PopulationOutOfSchoolApp.get('/years_of_study', (req, res, next) => {
    req.result = []

    for (let i = 0; i < 17; i++) {
        req.result.push({
            id: i, name: id2str.yearsOfStudy(i)
        });
    }

    req.result.push({id: 99, name: id2str.yearsOfStudy(99)});
    next();
}, response('years_of_study'));

PopulationOutOfSchoolApp.get('/instruction_level', (req, res, next) => {
    req.result = []

    for (let i = 1; i < 8; i++) {
        req.result.push({
            id: i, name: id2str.instructionLevel(i)
        });
    }
    req.result.push({id: 99, name: id2str.instructionLevel(99)});
    next();
}, response('instruction_level'));

PopulationOutOfSchoolApp.get('/new_pnad_adm_dependency', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 3; i++) {
        req.result.push({
            id: i, name: id2str.newPnadAdmDependency(i)
        });
    }
    req.result.push({id: 99, name: id2str.newPnadAdmDependency(99)});
    next();
}, response('new_pnad_adm_dependency'));

PopulationOutOfSchoolApp.get('/region', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 6; i++) {
        req.result.push({
            id: i, name: id2str.regionCode(i)
        });
    }

    next();
}, response('region'));

PopulationOutOfSchoolApp.get('/cap_code', (req, res, next) => {
    req.result = []
    for (let i = 11; i < 54; i++) {
        if (id2str.capitalCode(i) !== 'Não informado') {
        req.result.push({
            id: i, name: id2str.capitalCode(i)
        });
        }
    }
    req.result.push({id: 99, name: id2str.capitalCode(99)});

    next();
}, response('cap_code'));

PopulationOutOfSchoolApp.get('/metro_code', (req, res, next) => {
    req.result = []
    for (let i = 13; i < 53; i++) {
        if (id2str.metroCode(i) !== 'Não informado') {
        req.result.push({
            id: i, name: id2str.metroCode(i)
        });
        }
    }
    req.result.push({id: 99, name: id2str.metroCode(99)});

    next();
}, response('metro_code'));

PopulationOutOfSchoolApp.get('/attended_modality', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 16; i++) {
        req.result.push({
            id: i, name: id2str.attendedModality(i)
        });
    }
    // Remove the option with id equals 10 => This option exists in the database, a better solution to this would be remove the option from the database
    req.result.splice(req.result.findIndex((item) => item.id === 10), 1);
    req.result.push({id: 99, name: id2str.attendedModality(99)});
    next();
}, response('attended_modality'));

PopulationOutOfSchoolApp.get('/income_range', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 8; i++) {
        req.result.push({
            id: i, name: id2str.incomeRange(i)
        });
    }
    req.result.push({id: 9, name: id2str.incomeRange(9)});
    next();
}, response('income_range'));

PopulationOutOfSchoolApp.get('/attends_school', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 3; i++) {
        req.result.push({
            id: i, name: id2str.attendsSchool(i)
        });
    }
    next();
}, response('attends_school'));

PopulationOutOfSchoolApp.get('/gender', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 3; i++) {
        req.result.push({
            id: i, name: id2str.gender(i)
        });
    }
    next();
}, response('gender'));

PopulationOutOfSchoolApp.get('/new_pnad_ethnic_group', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 6; i++) {
        req.result.push({
            id: i, name: id2str.ethnicGroupNewPnad(i)
        });
    }
    req.result.push({id: 9, name: id2str.ethnicGroupNewPnad(9)});
    next();
}, response('new_pnad_ethnic_group'));

PopulationOutOfSchoolApp.get('/bolsa_familia', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 3; i++) {
        req.result.push({
            id: i, name: id2str.attendsSchool(i)
        });
    }
    req.result.push({id: 9, name: id2str.attendsSchool(9)});
    next();
}, response('bolsa_familia'));

PopulationOutOfSchoolApp.get('/modality', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 4; i++) {
        req.result.push({
            id: i, name: id2str.modality(i)
        });
    }
    req.result.push({id: 99, name: id2str.modality(99)});
    next();
}, response('modality'));

PopulationOutOfSchoolApp.get('/modality_shift', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 8; i++) {
        req.result.push({
            id: i, name: id2str.modalityShift(i)
        });
    }
    req.result.push({id: 9, name: id2str.modalityShift(9)});
    req.result.push({id: 99, name: id2str.modalityShift(99)});
    next();
}, response('modality_shift'));

PopulationOutOfSchoolApp.get('/state', (req, res, next) => {
    req.result = []
    for (let i = 11; i < 54; i++) {
        if (id2str.stateName(i) !== 'Não declarada') {
            req.result.push({
                id: i, name: id2str.stateName(i)
            });
        }
    }
    req.result.push({id: 99, name: id2str.stateName(99)});

    next();
}, response('state'));

PopulationOutOfSchoolApp.get('/age_range_pop_school', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 8; i++) {
        req.result.push({
            id: i, name: id2str.ageRangePopSchool(i)
        });
    }

    next();
}, response('age_range_pop_school'));


rqf.addField({
    name: 'filter',
    field: false,
    where: true
}).addField({
    name: 'dims',
    field: true,
    where: false
}).addValue({
    name: 'id',
    table: 'pnad_novo',
    tableField: 'id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    }
}).addValue({
    name: 'state',
    table: 'estado',
    tableField: ['id', 'nome'],
    resultField: ['state_id', 'state_nome'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id',
    },
    join: {
        primary: 'id',
        foreign: 'cod_uf',
        foreignTable: 'pnad_novo'
    }
}).addValue({
    name: 'state_not',
    table: 'estado',
    tableField: ['nome', 'id'],
    resultField: ['state_name', 'state_id'],
    where: {
        relation: '<>',
        type: 'integer',
        field: 'cod_uf',
        table: 'pnad_novo'
    },
    join: {
        primary: 'id',
        foreign: 'cod_uf',
        foreignTable: 'pnad_novo'
    }
}).addValue({
    name: 'bolsa_familia',
    table: 'pnad_novo',
    tableField: 'recebeu_rendimentos_de_programa_bolsa_familia',
    resultField: 'bolsa_familia_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'recebeu_rendimentos_de_programa_bolsa_familia'
    }
}).addValue({
    name: 'new_pnad_ethnic_group',
    table: 'pnad_novo',
    tableField: 'cor_raca',
    resultField: 'new_pnad_ethnic_group_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cor_raca'
    }
}).addValue({
    name: 'age_range_pop_school',
    table: 'pnad_novo',
    tableField: 'faixa_etaria_pop_out',
    resultField: 'age_range_pop_school_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'faixa_etaria_pop_out'
    }
}).addValue({
    name: 'income_range',
    table: 'pnad_novo',
    tableField: 'faixa_rendimento_aux',
    resultField: 'income_range_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'faixa_rendimento_aux'
    }
}).addValue({
    name: 'gender',
    table: 'pnad_novo',
    tableField: 'sexo',
    resultField: 'gender_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'sexo'
    }
}).addValue({
    name: 'cap_code',
    table: 'pnad_novo',
    tableField: 'cod_cap',
    resultField: 'cap_code_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_cap'
    }
}).addValue({
    name: 'region',
    table: 'pnad_novo',
    tableField: 'cod_regiao',
    resultField: 'region_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_regiao'
    }
}).addValue({
    name: 'metro_code',
    table: 'pnad_novo',
    tableField: 'cod_rm_ride',
    resultField: 'metro_code_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_rm_ride'
    }
}).addValue({
    name: 'min_year',
    table: 'pnad_novo',
    tableField: 'ano_ref',
    resultField: 'year',
    where: {
        relation: '>=',
        type: 'integer',
        field: 'ano_ref'
    }
}).addValue({
    name: 'max_year',
    table: 'pnad_novo',
    tableField: '',
    resultField: 'year',
    where: {
        relation: '<=',
        type: 'integer',
        field: 'ano_ref'
    }
}).addValue({
    name: 'location',
    table: 'pnad_novo',
    tableField: 'situacao_domicilio',
    resultField: 'location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'situacao_domicilio'
    }
});

function matchQueries(queryPartial, queryTotal) {
    let match = [];
    queryTotal.forEach((result) => {
        let newObj = {};
        let keys = Object.keys(result);
        keys.forEach((key) => {
            newObj[key] = result[key];
        });
        let index = keys.indexOf('total');
        if(index > -1) keys.splice(index, 1);
        let objMatch = null;

        for(let i = 0; i < queryPartial.length; ++i) {
            let partial = queryPartial[i];
            let foundMatch = true;
            for(let j = 0; j < keys.length; ++j) {
                let key = keys[j];
                if(partial[key] !== result[key]) {
                    foundMatch = false;
                    break;
                }
            }
            if(foundMatch) {
                objMatch = partial;
                break;
            }
        }

        if(objMatch) {
            newObj.denominator = result.total;
            newObj.pop_total = objMatch.total;
            newObj.total = Math.round((objMatch.total / result.total) * 10000) / 100;
            match.push(newObj);
        }
    }
)

    console.log(match)

    return match;
}

PopulationOutOfSchoolApp.get('/', rqf.parse(), rqf.build(),  (req, res, next) => {
    if ("age_range_pop_school" in req.filter || "age_range_pop_school" in req.dims) {
        // As we will need to do two requests, they'll be stored in a list
        req.querySet = []
        
        // Create an object that will store the first request (the sum of all people that attend school)
        let attends_school = req.sql.clone();
        attends_school.from('pnad_novo')
        .field('round(sum(pnad_novo.peso_domicilio_pessoas_com_cal), 0)', 'total')
        .field('pnad_novo.faixa_etaria_pop_out')
        .field('pnad_novo.ano_ref', 'year')
        .where('pnad_novo.ano_ref >= 2019')
        .where('pnad_novo.frequenta_escola = 2')
        .where('(pnad_novo.faixa_etaria_pop_out IN (1,2)) OR (pnad_novo.faixa_etaria_pop_out = 3 AND pnad_novo.nivel_de_instruc_mais_elevad_para_o_fundam_com_duracao_9_anos <= 2) OR \
        (pnad_novo.faixa_etaria_pop_out = 4 AND pnad_novo.nivel_de_instruc_mais_elevad_para_o_fundam_com_duracao_9_anos <= 3) OR \
        (pnad_novo.faixa_etaria_pop_out >= 5 AND pnad_novo.nivel_de_instruc_mais_elevad_para_o_fundam_com_duracao_9_anos <= 4)')
        .group('pnad_novo.ano_ref')
        .group('pnad_novo.faixa_etaria_pop_out')
        .order('pnad_novo.ano_ref')
        .order('pnad_novo.faixa_etaria_pop_out')
        req.querySet.push(attends_school);
        console.log(attends_school.toString());


        // Then, the second object is created and stores the sum of all people without filters
        let full_population = req.sql.clone();
        full_population.from('pnad_novo')
        .field('round(sum(pnad_novo.peso_domicilio_pessoas_com_cal), 0)', 'total')
        .field('pnad_novo.faixa_etaria_pop_out')
        .field('pnad_novo.ano_ref', 'year')
        .where('pnad_novo.ano_ref >= 2019')
        .group('pnad_novo.ano_ref')
        .group('pnad_novo.faixa_etaria_pop_out')
        .order('pnad_novo.ano_ref')
        .order('pnad_novo.faixa_etaria_pop_out')
        req.querySet.push(full_population);
    }

    next();
}, multiQuery, (req, res, next) => {
    if ("age_range_pop_school" in req.filter || "age_range_pop_school" in req.dims) {
        // The multiple requests are made. Then we need to calculate the percetange. So the function
        // below is used
        let newObj = matchQueries(req.result[0], req.result[1]);
        req.result = newObj;
        //console.log(attends_school.toString());
    } else {
        res.status(400);
        next({
            status: 400,
            message: 'Wrong/No filter specified'
        });
    }
    next();
}, id2str.transform(false), response('populationOutOfSchool'));

module.exports = PopulationOutOfSchoolApp;
