/*
Copyright (C) 2016 Centro de Computacao Cientifica e Software Livre
Departamento de Informatica - Universidade Federal do Parana - C3SL/UFPR

This file is part of simcaq-node.

simcaq-node is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

simcaq-node is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with simcaq-node.  If not, see <https://www.gnu.org/licenses/>.
*/

const express = require('express');

const rateSchoolApp = express.Router();

const libs = `${process.cwd()}/libs`;

const log = require(`${libs}/log`)(module);

const squel = require('squel');

const query = require(`${libs}/middlewares/query`).query;

const multiQuery = require(`${libs}/middlewares/multiQuery`);

const response = require(`${libs}/middlewares/response`);

const ReqQueryFields = require(`${libs}/middlewares/reqQueryFields`);

const id2str = require(`${libs}/middlewares/id2str`);

const config = require(`${libs}/config`);

const cache = require('apicache').options({ debug: config.debug, statusCodes: {include: [200]} }).middleware;

rateSchoolApp.use(cache('15 day'));

let rqf = new ReqQueryFields();

// Complete range of the enrollments dataset.
// Returns a tuple of start and ending years of the complete enrollments dataset.
rateSchoolApp.get('/year_range', (req, res, next) => {
    req.sql.from('pnad')
    .field('MIN(pnad.ano_censo)', 'start_year')
    .field('MAX(pnad.ano_censo)', 'end_year');
    next();
}, query, response('range'));

rateSchoolApp.get('/years', (req, res, next) => {
    req.sql.from('pnad')
    .field('DISTINCT pnad.ano_censo', 'year');
    next();
}, query, response('years'));

rateSchoolApp.get('/source', (req, res, next) => {
    req.sql.from('fonte')
    .field('fonte', 'source')
    .where('tabela = \'pnad\'');
    next();
}, query, response('source'));

rateSchoolApp.get('/ethnic_group_pnad', (req, res, next) => {
    req.result = [
        {id: 0, name: 'Indígena'},
        {id: 1, name: 'Branca e amarela'},
        {id: 2, name: 'Preta e parda'},
        {id: 9, name: 'Sem declaração'}
    ];
    next();
}, response('ethnic_group_pnad'));

rateSchoolApp.get('/age_range', (req, res, next) => {
    req.result = [
        {id: 1, name: '0 a 3 anos'},
        {id: 2, name: '4 a 5 anos'},
        {id: 3, name: '6 a 10 anos'},
        {id: 4, name: '11 a 14 anos'},
        {id: 5, name: '15 a 17 anos'},
        {id: 6, name: '18 a 24 anos'}
    ];
    next();
}, response('age_range'));

rateSchoolApp.get('/gender', (req, res, next) => {
    req.result = [
        {id: 1, name: 'Masculino'},
        {id: 2, name: 'Feminino'}
    ];
    next();
}, response('gender'));

rateSchoolApp.get('/location', (req, res, next) => {
    req.result = [
		{id: 1, name: 'Urbana'},
		{id: 2, name: 'Rural'}
	];
    next();
}, response('location'));

rateSchoolApp.get('/fifth_household_income', (req, res, next) => {
    req.result = [
		{id: 1, name: '20% menores'},
        {id: 2, name: '2o quinto'},
        {id: 3, name: '3o quinto'},
        {id: 4, name: '4o quinto'},
		{id: 5, name: '20% maiores'},
        {id: -1, name: 'Sem declaração'}
	];
    next();
},response('fifth_household_income'));

rateSchoolApp.get('/extremes_household_income', (req, res, next) => {
    req.result = [
		{id: 1, name: '10% menores'},
        {id: 2, name: '10% maiores'},
        {id: -1, name: 'Sem declaração'}
	];
    next();
}, response('extremes_household_income'));

rqf.addField({
    name: 'filter',
    field: false,
    where: true
}).addField({
    name: 'dims',
    field: true,
    where: false
}).addValue({
    name: 'region',
    table: 'regiao',
    tableField: ['nome', 'id'],
    resultField: ['region_name', 'region_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'regiao_id',
        foreignTable: 'pnad'
    }
}).addValue({
    name: 'state',
    table: 'estado',
    tableField: ['nome', 'id'],
    resultField: ['state_name', 'state_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'estado_id',
        foreignTable: 'pnad'
    }
}).addValue({
    name: 'ethnic_group_pnad',
    table: 'pnad',
    tableField: 'cor_raca',
    resultField: 'ethnic_group_pnad_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cor_raca'
    }
}).addValue({
    name: 'min_year',
    table: 'pnad',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '>=',
        type: 'integer',
        table: 'pnad',
        field: 'ano_censo'
    }
}).addValue({
    name: 'max_year',
    table: 'pnad',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '<=',
        type: 'integer',
        table: 'pnad',
        field: 'ano_censo'
    }
}).addValue({
    name: 'age_range',
    table: 'pnad',
    tableField: 'faixa_etaria_31_03',
    resultField: 'age_range_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'faixa_etaria_31_03'
    }
}).addValue({
    name: 'gender',
    table: 'pnad',
    tableField: 'sexo',
    resultField: 'gender_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'sexo'
    }
}).addValue({
    name: 'location',
    table: 'pnad',
    tableField: 'localizacao_id',
    resultField: 'location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localizacao_id'
    }
}).addValue({
    name: 'extremes_household_income',
    table: 'pnad',
    tableField: 'extremos_nivel_rendimento',
    resultField: 'extremes_household_income_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'extremos_nivel_rendimento'
    }
}).addValue({
    name: 'fifth_household_income',
    table: 'pnad',
    tableField: 'quintil_nivel_rendimento',
    resultField: 'fifth_household_income_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'quintil_nivel_rendimento'
    }
});

function matchQueries(queryTotal, queryPartial) {
    let match = [];
    queryTotal.forEach((result) => {
        let newObj = {};
        let keys = Object.keys(result);
        keys.forEach((key) => {
            newObj[key] = result[key];
        });
        // console.log('NEW OBJ');
        // console.log(newObj);
        let index = keys.indexOf('total');
        if(index > -1) keys.splice(index, 1);
        let objMatch = null;

        for(let i = 0; i < queryPartial.length; ++i) {
            let partial = queryPartial[i];
            let foundMatch = true;
            for(let j = 0; j < keys.length; ++j) {
                let key = keys[j];
                if(partial[key] !== result[key]) {
                    foundMatch = false;
                    break;
                }
            }
            if(foundMatch) {
                objMatch = partial;
                break;
            }
        }

        if(objMatch) {
            // console.log(objMatch);
            newObj.denominator = result.total;
            newObj.partial = objMatch.total;
            newObj.total = (objMatch.total / result.total) * 100;
            match.push(newObj);
        }
    });

    return match;
}

rateSchoolApp.get('/', rqf.parse(), rqf.build(), (req, res, next) => {
    req.querySet = [];
    req.queryIndex = {};

    log.debug(req.sql.toParam());
    if ("age_range" in req.filter || "age_range" in req.dims) {
        let freq_total = req.sql.clone();
        freq_total.field('sum(pnad.peso)', 'total')
        .field('pnad.ano_censo','year')
        .from('pnad')
        .group('pnad.ano_censo')
        .order('pnad.ano_censo')
        .where('pnad.faixa_etaria_31_03 < 7')
        req.queryIndex.freq_total = req.querySet.push(freq_total) - 1;

        let freq_nursery = req.sql.clone();
        freq_nursery.field('sum(pnad.peso)', 'total')
        .field('pnad.ano_censo','year')
        .from('pnad')
        .group('pnad.ano_censo')
        .order('pnad.ano_censo')
        .where('pnad.frequenta_escola_creche = 2')
        .where('pnad.faixa_etaria_31_03 < 7')
        req.queryIndex.freq_nursery = req.querySet.push(freq_nursery) - 1;
    }
     next();
}, multiQuery, (req, res, next) => {
    if ("age_range" in req.filter || "age_range" in req.dims) {
        log.debug(req.result[req.queryIndex.freq_total]);
        log.debug(req.result[req.queryIndex.freq_nursery])
        let school_attendance_rate = matchQueries(req.result[req.queryIndex.freq_total], req.result[req.queryIndex.freq_nursery]);
        req.result = school_attendance_rate;
    } else {
        res.status(400);
        next({
            status: 400,
            message: 'Wrong/No filter specified'
        });
    }
    log.debug(req.result)
    next();
}, id2str.transform(false), response('rateSchool'));

module.exports = rateSchoolApp;
