/*
Copyright (C) 2016 Centro de Computacao Cientifica e Software Livre
Departamento de Informatica - Universidade Federal do Parana - C3SL/UFPR

This file is part of simcaq-node.

simcaq-node is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

simcaq-node is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with simcaq-node.  If not, see <https://www.gnu.org/licenses/>.
*/

const express = require('express');

const auxiliarApp = express.Router();

const libs = `${process.cwd()}/libs`;

const log = require(`${libs}/log`)(module);

const squel = require('squel');

const query = require(`${libs}/middlewares/query`).query;

const response = require(`${libs}/middlewares/response`);

const ReqQueryFields = require(`${libs}/middlewares/reqQueryFields`);

const id2str = require(`${libs}/middlewares/id2str`);

const config = require(`${libs}/config`);

const addMissing = require(`${libs}/middlewares/addMissing`);

const cache = require('apicache').options({ debug: config.debug, statusCodes: {include: [200]}  }).middleware;

let rqf = new ReqQueryFields();

auxiliarApp.use(cache('15 day'));

auxiliarApp.get('/year_range', (req, res, next) => {
    req.sql.from('docente')
    .field('MIN(docente.ano_censo)', 'start_year')
    .field('MAX(docente.ano_censo)', 'end_year');
    next();
}, query, response('range'));

auxiliarApp.get('/years', (req, res, next) => {
    req.sql.from('docente').
    field('DISTINCT docente.ano_censo', 'year')
    next();
}, query, response('years'));

auxiliarApp.get('/source', (req, res, next) => {
    req.sql.from('fonte')
    .field('fonte', 'source')
    .where('tabela = \'docente\'');
    next();
}, query, response('source'))

auxiliarApp.get('/education_type', (req, res, next) => {
    req.sql.from('docente')
    .field('DISTINCT nivel_tipo_formacao', 'id')
    .order('id');
    next();
}, query, (req, res, next) => {
    req.result.forEach((result) => {
        result.name = id2str.educationType(result.id);
    });
    next();
}, response('education_type'));

auxiliarApp.get('/adm_dependency_detailed', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 7; ++i) {
        req.result.push({
            id: i,
            name: id2str.admDependencyPriv(i)
        });
    };
    next();
}, response('adm_dependency_detailed'));

auxiliarApp.get('/adm_dependency', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 4; ++i) {
        req.result.push({
            id: i,
            name: id2str.admDependency(i)
        });
    };
    next();
}, response('adm_dependency'));

auxiliarApp.get('/location', (req, res, next) => {
    req.result = [
        {id: 1, name: 'Urbana'},
        {id: 2, name: 'Rural'}
    ];
    next();
}, response('location'));

auxiliarApp.get('/diff_location', (req, res, next) => {
    req.result = [
        {id: 0, name: "A escola não está em localidade diferenciada"},
        {id: 1, name: "Área de assentamento"},
        {id: 2, name: "Terra indígena"},
        {id: 3, name: "Terra remanescente de quilombos"},
    ];
    next();
}, response('diff_location'));

auxiliarApp.get('/education_level_mod', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 12; ++i) {
        req.result.push({
            id: i,
            name: id2str.educationLevelMod(i)
        });
    }

    req.result.push({
        id: 99,
        name: id2str.educationLevelMod(99)
    });
    next();
}, response('education_level_mod'));

auxiliarApp.get('/gender', (req, res, next) => {
    req.result = [
        {id: 1, name: 'Masculino'},
        {id: 2, name: 'Feminino'}
    ];
    next();
}, response('gender'));

auxiliarApp.get('/ethnic_group', (req, res, next) => {
    req.result = [];
    for(let i = 0; i <=5; ++i) {
        req.result.push({
            id: i,
            name: id2str.ethnicGroup(i)
        });
    }
    next();
}, response('ethnic_group'));

rqf.addField({
    name: 'filter',
    field: false,
    where: true
}).addField({
    name: 'dims',
    field: true,
    where: false
}).addValue({
    name: 'adm_dependency',
    table: 'docente',
    tableField: 'dependencia_adm_id',
    resultField: 'adm_dependency_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'dependencia_adm_id'
    }
}).addValue({
    name: 'adm_dependency_detailed',
    table: 'docente',
    tableField: 'dependencia_adm_priv',
    resultField: 'adm_dependency_detailed_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'dependencia_adm_priv'
    }
}).addValue({
    name: 'education_type',
    table: 'docente',
    tableField: 'nivel_tipo_formacao',
    resultField: 'education_type_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'nivel_tipo_formacao'
    }
}).addValue({
    name: 'education_level_mod',
    table: 'docente',
    tableField: 'etapas_mod_ensino_segmento_id',
    resultField: 'education_level_mod_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'etapas_mod_ensino_segmento_id'
    }
}).addValue({
  name: 'region',
  table: 'regiao',
  tableField: ['nome', 'id'],
  resultField: ['region_name', 'region_id'],
  where: {
      relation: '=',
      type: 'integer',
      field: 'id'
  },
  join: {
      primary: 'id',
      foreign: 'escola_regiao_id',
      foreignTable: 'docente'
  }
}).addValue({
    name: 'state',
    table: 'estado',
    tableField: ['nome', 'id'],
    resultField: ['state_name', 'state_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'escola_estado_id',
        foreignTable: 'docente'
    }
}).addValue({
    name: 'rural_location',
    table: 'docente',
    tableField: 'localidade_area_rural',
    resultField: 'rural_location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localidade_area_rural'
    }
}).addValueToField({
    name: 'city',
    table: 'municipio',
    tableField: ['nome', 'id'],
    resultField: ['city_name', 'city_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'escola_municipio_id',
        foreignTable: 'docente'
    }
}, 'dims').addValueToField({
    name: 'city',
    table: 'municipio',
    tableField: 'nome',
    resultField: 'city_name',
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'escola_municipio_id',
        foreignTable: 'docente'
    }
}, 'filter').addValueToField({
    name: 'school',
    table: 'escola',
    tableField: ['nome_escola', 'id'],
    resultField: ['school_name', 'school_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: ['id', 'ano_censo'],
        foreign: ['escola_id', 'ano_censo'],
        foreignTable: 'docente'
    }
}, 'dims').addValueToField({
    name: 'school',
    table: 'escola',
    tableField: 'nome_escola',
    resultField: 'school_name',
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: ['id', 'ano_censo'],
        foreign: ['escola_id', 'ano_censo'],
        foreignTable: 'docente'
    }
}, 'filter').addValue({
    name: 'location',
    table: 'docente',
    tableField: 'localizacao_id',
    resultField: 'location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localizacao_id'
    }
}).addValue({
    name: 'diff_location',
    table: 'docente',
    tableField: 'localizacao_diferenciada_par',
    resultField: 'diff_location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localizacao_diferenciada_par'
    }
}).addValue({
    name: 'min_year',
    table: 'docente',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '>=',
        type: 'integer',
        field: 'ano_censo'
    }
}).addValue({
    name: 'max_year',
    table: 'docente',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '<=',
        type: 'integer',
        field: 'ano_censo'
    }
}).addValue({
    name: 'gender',
    table: 'docente',
    tableField: 'sexo',
    resultField: 'gender_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'sexo'
    }
}).addValue({
    name: 'ethnic_group',
    table: 'docente',
    tableField: 'cor_raca',
    resultField: 'ethnic_group_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cor_raca'
    }
});

// LDE
auxiliarApp.get('/', rqf.parse(), (req, res, next) => {
  req.sql.field('COUNT(DISTINCT docente.id_docente)', 'total')
  .field("'Brasil'", 'name')
  .field('docente.ano_censo', 'year')
  .from('docente')
  .join('turma', null, 'docente.turma_id=turma.id AND docente.ano_censo=turma.ano_censo')
  .group('docente.ano_censo')
  .order('docente.ano_censo')
  .where('(docente.tipo_docente = 2) AND \
    ((docente.tipo_turma_id >= 0 AND docente.tipo_turma_id <= 3 AND docente.tipo_turma_atendimento_id is NULL) \
    OR ((docente.tipo_turma_atendimento_id = 1 OR docente.tipo_turma_atendimento_id = 2) AND docente.tipo_turma_id is NULL)) \
    AND (docente.ano_censo <> 2009 or (docente.escola_estado_id <> 42 AND docente.escola_estado_id <> 43))');     // não devemos trazer SC em 2009.
  next();
}, rqf.build(), query, addMissing(rqf), id2str.transform(), response('auxiliar'));

// SimCAQ
auxiliarApp.get('/count', rqf.parse(), (req, res, next) => {
    req.sql.field('COUNT(DISTINCT docente.id_docente)', 'total')
   .field("'Brasil'", 'name')
   .field('docente.ano_censo', 'year')
   .from('docente')
   .group('docente.ano_censo')
   .order('docente.ano_censo')
   .where('(tipo_turma_id <= 3) AND (tipo_docente=2) AND (dependencia_adm_id = 1 OR dependencia_adm_id = 2 OR dependencia_adm_id = 3) AND  \
         (etapas_mod_ensino_segmento_id=1 OR etapas_mod_ensino_segmento_id=2 OR etapas_mod_ensino_segmento_id=4 OR \
         etapas_mod_ensino_segmento_id=5 OR etapas_mod_ensino_segmento_id=6 OR etapas_mod_ensino_segmento_id=8 OR etapas_mod_ensino_segmento_id=9)');

  next();
}, rqf.build(), query, addMissing(rqf), id2str.transform(), response('auxiliar'));

module.exports = auxiliarApp;
