/*
Copyright (C) 2024 Centro de Computacao Cientifica e Software Livre
Departamento de Informatica - Universidade Federal do Parana - C3SL/UFPR

This file is part of simcaq-node.

simcaq-node is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

simcaq-node is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with simcaq-node.  If not, see <https://www.gnu.org/licenses/>.
*/

const express = require('express');

const NivelInstrucao = express.Router();

const libs = `${process.cwd()}/libs`;

const squel = require('squel');

const query = require(`${libs}/middlewares/query`).query;

const response = require(`${libs}/middlewares/response`);

const ReqQueryFields = require(`${libs}/middlewares/reqQueryFields`);

const id2str = require(`${libs}/middlewares/id2str`);

const config = require(`${libs}/config`); 

const cache = require('apicache').options({ debug: config.debug, statusCodes: {include: [200]} }).middleware;

let rqf = new ReqQueryFields();

NivelInstrucao.use(cache('15 day'));

NivelInstrucao.get('/years', (req, res, next) => {
    req.sql.from('pnad_novo')
    .field('DISTINCT pnad_novo.ano_ref', 'year')
    next();
}, query, response('years'));

NivelInstrucao.get('/instruction_level', (req, res, next) => {
    req.result = []

    for (let i = 1; i < 8; i++) {
        req.result.push({
            id: i, name: id2str.instructionLevel(i)
        });

    }
    req.result.push({id: 99, name: id2str.instructionLevel(99)});
    next();
}, response('instruction_level'));

NivelInstrucao.get('/age_range_all', (req, res, next) => {
    req.result = []

    for (let i = 1; i < 9; i++) {
        req.result.push({
            id: i, name: id2str.ageRangeAll(i)
        });

    }
    req.result.push({id: 99, name: id2str.ageRangeAll(99)});
    next();
}, response('age_range_all'));

NivelInstrucao.get('/gender', (req, res, next) => {
    req.result = []

    for (let i = 1; i < 3; i++) {
        req.result.push({
            id: i, name: id2str.gender(i)
        });
    }
    next();
}, response('gender'));

NivelInstrucao.get('/bolsa_familia', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 3; i++) {
        req.result.push({
            id: i, name: id2str.attendsSchool(i)
        });
    }
    req.result.push({id: 9, name: id2str.attendsSchool(9)});
    next();
}, response('bolsa_familia'));

NivelInstrucao.get('/location', (req, res, next) => {
    req.result = []
    for (let i = 1; i <= 2; i++) {
        req.result.push({
            id: i, name: id2str.location(i)
        });
    }

    next();
}, response('location'));

NivelInstrucao.get('/ethnic_group', (req, res, next) => {
    req.result = [];
    for(let i = 0; i <=5; ++i) {
        req.result.push({
            id: i,
            name: id2str.ethnicGroup(i)
        });
    }
    next();
}, response('ethnic_group'));


rqf.addField({
    name: 'filter',
    field: false,
    where: true
}).addField({
    name: 'dims',
    field: true,
    where: false
}).addValue({
    name: 'id',
    table: 'pnad_novo',
    tableField: 'id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    }
}).addValue({
    name: 'bolsa_familia',
    table: 'pnad_novo',
    tableField: 'recebeu_rendimentos_de_programa_bolsa_familia',
    resultField: 'bolsa_familia_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'recebeu_rendimentos_de_programa_bolsa_familia'
    }
}).addValue({
    name: 'cap_code',
    table: 'pnad_novo',
    tableField: 'cod_cap',
    resultField: 'cap_code_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_cap'
    }
}).addValue({
    name: 'new_pnad_ethnic_group',
    table: 'pnad_novo',
    tableField: 'cor_raca',
    resultField: 'new_pnad_ethnic_group_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cor_raca'
    }
}).addValue({
    name: 'age_range_all',
    table: 'pnad_novo',
    tableField: 'faixa_etaria',
    resultField: 'age_range_all_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'faixa_etaria'
    }
}).addValue({
    name: 'income_range',
    table: 'pnad_novo',
    tableField: 'faixa_rendimento_aux',
    resultField: 'income_range_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'faixa_rendimento_aux'
    }
}).addValue({
    name: 'region',
    table: 'pnad_novo',
    tableField: 'cod_regiao',
    resultField: 'region_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_regiao'
    }
}).addValue({
    name: 'metro_code',
    table: 'pnad_novo',
    tableField: 'cod_rm_ride',
    resultField: 'metro_code_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_rm_ride'
    }
}).addValue({
    name: 'gender',
    table: 'pnad_novo',
    tableField: 'sexo',
    resultField: 'gender_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'sexo'
    }
}).addValue({
    name: 'state',
    table: 'estado',
    tableField: ['id', 'nome'],
    resultField: ['state_id', 'state_nome'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id',
    },
    join: {
        primary: 'id',
        foreign: 'cod_uf',
        foreignTable: 'pnad_novo'
    }
}).addValue({
    name: 'instruction_level',
    table: 'pnad_novo',
    tableField: 'nivel_de_instrucao',
    resultField: 'instruction_level_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_uf'
    }
}).addValue({
    name: 'min_year',
    table: 'pnad_novo',
    tableField: 'ano_ref',
    resultField: 'year',
    where: {
        relation: '>=',
        type: 'integer',
        field: 'ano_ref'
    }
}).addValue({
    name: 'max_year',
    table: 'pnad_novo',
    tableField: '',
    resultField: 'year',
    where: {
        relation: '<=',
        type: 'integer',
        field: 'ano_ref'
    }
}).addValue({
    name: 'bolsa_familia',
    table: 'pnad_novo',
    tableField: 'recebeu_rendimentos_de_programa_bolsa_familia',
    resultField: 'bolsa_familia_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'recebeu_rendimentos_de_programa_bolsa_familia'
    }
}).addValue({
    name: 'location',
    table: 'pnad_novo',
    tableField: 'situacao_domicilio',
    resultField: 'location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'situacao_domicilio'
    }
})

NivelInstrucao.get('/', rqf.parse(), rqf.build(),  (req, res, next) => {
    if ("instruction_level" in req.filter || "instruction_level" in req.dims) {
        // The multiple requests are made. Then we need to calculate the percetange. So the function
        // below is used
        req.sql.from('pnad_novo')
        .field('nivel_de_instrucao', 'nivel')
        .field('round(sum(peso_domicilio_pessoas_com_cal), 0)', 'total')
        .field('ano_ref', 'year')
        .where('nivel_de_instrucao <> 99')
        .where('faixa_etaria <> 99')
        .group('pnad_novo.ano_ref')
        .group('pnad_novo.nivel_de_instrucao')
        console.log(req.sql.toString())
    }
    else {
        res.status(400);
        next({
            status: 400,
            message: 'Wrong/No filter specified'
        });
    }
    next();
}, query, id2str.transform(false), response('pnad_novo'));

module.exports = NivelInstrucao;
