/*
Copyright (C) 2024 Centro de Computacao Cientifica e Software Livre
Departamento de Informatica - Universidade Federal do Parana - C3SL/UFPR

This file is part of simcaq-node.

simcaq-node is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

simcaq-node is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with simcaq-node.  If not, see <https://www.gnu.org/licenses/>.
*/

const express = require('express');

const PnadNovoApp = express.Router();

const libs = `${process.cwd()}/libs`;

const squel = require('squel');

const query = require(`${libs}/middlewares/query`).query;

const response = require(`${libs}/middlewares/response`);

const ReqQueryFields = require(`${libs}/middlewares/reqQueryFields`);

const id2str = require(`${libs}/middlewares/id2str`);

const config = require(`${libs}/config`); 

const cache = require('apicache').options({ debug: config.debug, statusCodes: {include: [200]} }).middleware;

let rqf = new ReqQueryFields();

PnadNovoApp.use(cache('15 day'));

PnadNovoApp.get('/years', (req, res, next) => {
    req.sql.from('pnad_novo')
    .field('DISTINCT pnad_novo.ano_ref', 'year')
    next();
}, query, response('years'));

PnadNovoApp.get('/illiteracy', (req, res, next) => {
    req.result = []

    for (let i = 0; i < 2; i++) {
        req.result.push({
            id: i, name: id2str.illiteracy(i)
        });

    }
    req.result.push({id: 9, name: id2str.illiteracy(9)});
    next();
}, response('illiteracy'));

PnadNovoApp.get('/years_of_study', (req, res, next) => {
    req.result = []

    for (let i = 0; i < 17; i++) {
        req.result.push({
            id: i, name: id2str.yearsOfStudy(i)
        });
    }

    req.result.push({id: 99, name: id2str.yearsOfStudy(99)});
    next();
}, response('years_of_study'));

PnadNovoApp.get('/instruction_level', (req, res, next) => {
    req.result = []

    for (let i = 1; i < 8; i++) {
        req.result.push({
            id: i, name: id2str.instructionLevel(i)
        });
    }
    req.result.push({id: 99, name: id2str.instructionLevel(99)});
    next();
}, response('instruction_level'));

PnadNovoApp.get('/new_pnad_adm_dependency', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 3; i++) {
        req.result.push({
            id: i, name: id2str.newPnadAdmDependency(i)
        });
    }
    req.result.push({id: 99, name: id2str.newPnadAdmDependency(99)});
    next();
}, response('new_pnad_adm_dependency'));

PnadNovoApp.get('/region', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 6; i++) {
        req.result.push({
            id: i, name: id2str.regionCode(i)
        });
    }

    next();
}, response('region'));

PnadNovoApp.get('/cap_code', (req, res, next) => {
    req.result = []
    for (let i = 11; i < 54; i++) {
        if (id2str.capitalCode(i) !== 'Não informado') {
        req.result.push({
            id: i, name: id2str.capitalCode(i)
        });
        }
    }
    req.result.push({id: 99, name: id2str.capitalCode(99)});

    next();
}, response('cap_code'));

PnadNovoApp.get('/metro_code', (req, res, next) => {
    req.result = []
    for (let i = 13; i < 53; i++) {
        if (id2str.metroCode(i) !== 'Não informado') {
        req.result.push({
            id: i, name: id2str.metroCode(i)
        });
        }
    }
    req.result.push({id: 99, name: id2str.metroCode(99)});

    next();
}, response('metro_code'));

PnadNovoApp.get('/attended_modality', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 16; i++) {
        req.result.push({
            id: i, name: id2str.attendedModality(i)
        });
    }
    // Remove the option with id equals 10 => This option exists in the database, a better solution to this would be remove the option from the database
    req.result.splice(req.result.findIndex((item) => item.id === 10), 1);
    req.result.push({id: 99, name: id2str.attendedModality(99)});
    next();
}, response('attended_modality'));

PnadNovoApp.get('/income_range', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 8; i++) {
        req.result.push({
            id: i, name: id2str.incomeRange(i)
        });
    }
    req.result.push({id: 9, name: id2str.incomeRange(9)});
    next();
}, response('income_range'));

PnadNovoApp.get('/attends_school', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 3; i++) {
        req.result.push({
            id: i, name: id2str.attendsSchool(i)
        });
    }
    next();
}, response('attends_school'));

PnadNovoApp.get('/gender', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 3; i++) {
        req.result.push({
            id: i, name: id2str.gender(i)
        });
    }
    next();
}, response('gender'));

PnadNovoApp.get('/new_pnad_ethnic_group', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 6; i++) {
        req.result.push({
            id: i, name: id2str.ethnicGroupNewPnad(i)
        });
    }
    req.result.push({id: 9, name: id2str.ethnicGroupNewPnad(9)});
    next();
}, response('new_pnad_ethnic_group'));

PnadNovoApp.get('/bolsa_familia', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 3; i++) {
        req.result.push({
            id: i, name: id2str.attendsSchool(i)
        });
    }
    req.result.push({id: 9, name: id2str.attendsSchool(9)});
    next();
}, response('bolsa_familia'));

PnadNovoApp.get('/modality', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 4; i++) {
        req.result.push({
            id: i, name: id2str.modality(i)
        });
    }
    req.result.push({id: 99, name: id2str.modality(99)});
    next();
}, response('modality'));

PnadNovoApp.get('/modality_shift', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 8; i++) {
        req.result.push({
            id: i, name: id2str.modalityShift(i)
        });
    }
    req.result.push({id: 9, name: id2str.modalityShift(9)});
    req.result.push({id: 99, name: id2str.modalityShift(99)});
    next();
}, response('modality_shift'));

PnadNovoApp.get('/state', (req, res, next) => {
    req.result = []
    for (let i = 11; i < 54; i++) {
        if (id2str.stateName(i) !== 'Não declarada') {
            req.result.push({
                id: i, name: id2str.stateName(i)
            });
        }
    }
    req.result.push({id: 99, name: id2str.stateName(99)});

    next();
}, response('state'));

PnadNovoApp.get('/age_range_all', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 12; i++) {
        req.result.push({
            id: i, name: id2str.ageRangeAll(i)
        });
    }

    next();
}, response('age_range_all'));


rqf.addField({
    name: 'filter',
    field: false,
    where: true
}).addField({
    name: 'dims',
    field: true,
    where: false
}).addValue({
    name: 'id',
    table: 'pnad_novo',
    tableField: 'id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    }
}).addValue({
    name: 'state',
    table: 'estado',
    tableField: ['id', 'nome'],
    resultField: ['state_id', 'state_nome'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id',
    },
    join: {
        primary: 'id',
        foreign: 'cod_uf',
        foreignTable: 'pnad_novo'
    }
}).addValue({
    name: 'state_not',
    table: 'estado',
    tableField: ['nome', 'id'],
    resultField: ['state_name', 'state_id'],
    where: {
        relation: '<>',
        type: 'integer',
        field: 'cod_uf',
        table: 'pnad_novo'
    },
    join: {
        primary: 'id',
        foreign: 'cod_uf',
        foreignTable: 'pnad_novo'
    }
}).addValue({
    name: 'location',
    table: 'pnad_novo',
    tableField: 'situacao_domicilio',
    resultField: 'location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'situacao_domicilio'
    }
}).addValue({
    name: 'instruction_level',
    table: 'pnad_novo',
    tableField: 'nivel_de_instrucao',
    resultField: 'instruction_level_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'nivel_de_instrucao'
    }
}).addValue({
    name: 'new_pnad_adm_dependency',
    table: 'pnad_novo',
    tableField: 'dependencia_adm',
    resultField: 'new_pnad_adm_dependency_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'dependencia_adm'
    }
}).addValue({
    name: 'attends_school',
    table: 'pnad_novo',
    tableField: 'frequenta_escola',
    resultField: 'attends_school_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'frequenta_escola'
    }
}).addValue({
    name: 'modality',
    table: 'pnad_novo',
    tableField: 'modalidade',
    resultField: 'modality_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'modalidade'
    }
}).addValue({
    name: 'attended_modality',
    table: 'pnad_novo',
    tableField: 'nivel_etapa_modalidade_freq',
    resultField: 'attended_modality_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'nivel_etapa_modalidade_freq'
    }
}).addValue({
    name: 'illiteracy',
    table: 'pnad_novo',
    tableField: 'analfabetismo',
    resultField: 'illiteracy_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'analfabetismo'
    }
}).addValue({
    name: 'modality_shift',
    table: 'pnad_novo',
    tableField: 'turno_nivel_etapa',
    resultField: 'modality_shift_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'turno_nivel_etapa'
    }               
}).addValue({
    name: 'bolsa_familia',
    table: 'pnad_novo',
    tableField: 'recebeu_rendimentos_de_programa_bolsa_familia',
    resultField: 'bolsa_familia_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'recebeu_rendimentos_de_programa_bolsa_familia'
    }
}).addValue({
    name: 'new_pnad_ethnic_group',
    table: 'pnad_novo',
    tableField: 'cor_raca',
    resultField: 'new_pnad_ethnic_group_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cor_raca'
    }
}).addValue({
    name: 'age_range_all',
    table: 'pnad_novo',
    tableField: 'faixa_etaria',
    resultField: 'age_range_all_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'faixa_etaria'
    }
}).addValue({
    name: 'income_range',
    table: 'pnad_novo',
    tableField: 'faixa_rendimento_aux',
    resultField: 'income_range_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'faixa_rendimento_aux'
    }
}).addValue({
    name: 'gender',
    table: 'pnad_novo',
    tableField: 'sexo',
    resultField: 'gender_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'sexo'
    }
}).addValue({
    name: 'cap_code',
    table: 'pnad_novo',
    tableField: 'cod_cap',
    resultField: 'cap_code_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_cap'
    }
}).addValue({
    name: 'region',
    table: 'pnad_novo',
    tableField: 'cod_regiao',
    resultField: 'region_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_regiao'
    }
}).addValue({
    name: 'metro_code',
    table: 'pnad_novo',
    tableField: 'cod_rm_ride',
    resultField: 'metro_code_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_rm_ride'
    }
}).addValue({
    name: 'min_year',
    table: 'pnad_novo',
    tableField: 'ano_ref',
    resultField: 'year',
    where: {
        relation: '>=',
        type: 'integer',
        field: 'ano_ref'
    }
}).addValue({
    name: 'max_year',
    table: 'pnad_novo',
    tableField: 'ano_ref',
    resultField: 'year',
    where: {
        relation: '<=',
        type: 'integer',
        field: 'ano_ref'
    }
}).addField({
    name: 'search',
    field: false,
    where: true 
}).addValueToField({
    name: 'name',
    table: 'pnad_novo',
    tableField: 'nome',
    where: {
        relation: 'LIKE',
        type: 'string',
        field: 'nome'
    }
}, 'search').addValue({
    name: 'mesoregion',
    table: 'pnad_novo',
    tableField: 'mesorregiao_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'mesorregiao_id'
    }
}).addValue({
    name: 'microregion',
    table: 'pnad_novo',
    tableField: 'microrregiao_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'microrregiao_id'
    }
});

PnadNovoApp.get('/', rqf.parse(), rqf.build(),  (req, res, next) => {
    const dimensions = req.dims || {};
    
    let joinQuery = `
        pnad_novo.ano_ref = media.ano_ref
        ${dimensions.gender ? " AND media.gender_id = pnad_novo.sexo" : ""}
        ${dimensions.bolsa_familia ? "AND media.bolsa_familia_id = pnad_novo.recebeu_rendimentos_de_programa_bolsa_familia" : ""}
        ${dimensions.new_pnad_ethnic_group ? "AND media.new_pnad_ethnic_group_id = pnad_novo.cor_raca" : ""}
        ${dimensions.income_range ? "AND media.income_range_id = pnad_novo.faixa_rendimento_aux_tx" : ""}
        ${dimensions.age_range_all ? "AND media.age_range_all_id = pnad_novo.faixa_etaria" : ""}
        ${dimensions.location ? "AND media.location_id = pnad_novo.situacao_domicilio" : ""}
        ${dimensions.metro_code ? "AND media.metro_code_id = pnad_novo.cod_rm_ride" : ""}
        ${dimensions.cap_code ? "AND media.cap_code_id = pnad_novo.cod_cap" : ""}
        ${dimensions.region ? "AND media.region_id = pnad_novo.cod_regiao" : ""}
        ${dimensions.state ? "AND media.state_id = pnad_novo.cod_uf" : ""}
    `

    let mean = req.sql.clone()
    .from('pnad_novo')
    .field('pnad_novo.ano_ref', 'ano_ref')
    .field('SUM(pnad_novo.anos_de_estudo * pnad_novo.peso_domicilio_pessoas_com_cal) / SUM(pnad_novo.peso_domicilio_pessoas_com_cal)', 'media_anos_estudo')
    .where('pnad_novo.anos_de_estudo <> 99 and pnad_novo.faixa_etaria >= 6 and pnad_novo.ano_ref >= 2019')
    .group('pnad_novo.ano_ref')

    req.sql.from('pnad_novo')
    .field('pnad_novo.ano_ref', 'year')
    .field('media.media_anos_estudo', 'average')
    .field('SQRT(SUM(POWER(pnad_novo.anos_de_estudo - media.media_anos_estudo, 2) * pnad_novo.peso_domicilio_pessoas_com_cal) / SUM(pnad_novo.peso_domicilio_pessoas_com_cal))', 'std_dev')
    .join(mean, 'media', joinQuery)
    .where('pnad_novo.ano_ref >= 2019 AND pnad_novo.faixa_etaria >= 6 AND pnad_novo.anos_de_estudo <> 99')
    .group('pnad_novo.ano_ref')
    .group('media.media_anos_estudo')

    next();
}, query, id2str.transform(false), response('years_of_study'));

module.exports = PnadNovoApp;
